//****************************************************************************
// 
// jsxmlparser.js
// Scott Brause
// 2003-03-27
//
// Description:
//
// Contains the JSXMLParser Object constructor and member functions.
//
// The JSXMLParser Object, along with the TextFetcher.jar file and tinyxmldom.js,
// allow you to load an XML document from the server, and parse that document.
//
// The JSXMLParser's parse functions, parseString() and parseFile(), read in the 
// specified String or file, parse it into an XMLDoc Object, traverse the
// XMLDoc Object, firing SAX-like events as it goes, and then return a reference
// to the XMLDoc Object.
//
// The SAX-like events are caught by instantiating a JSXMLDocumentHandler Object,
// over-riding it's default handler functions, and passing it to the JSXMLParser
// Object's constructor.
//
// Dependencies: TextFetcher.jar - a Java applet that can read a text file from
//                                 the server
//
//               tinyxmldom.js - a JS file that contains code that implements
//                               an XML parser.
//
// Usage:
//
// Include this file in the 
//
//****************************************************************************

/* Sample Code

example.html:
-------------

<html><head><title>JSXMLParser Example</title>
<script src="tinyxmldom.js" language="JavaScript1.2"></script>
<script language="JavaScript" type="text/JavaScript">
var FILENAME = "myfile.xml"; // The xml file on the server.
var parser; // To hold the JSXMLParser Object.
var docHandler; // To hold the JSXMLDocumentHandler.

// Define an error handler
function errHandler(err)
{
	alert("errHandler: " + err);
}

// Define the document handler functions that will over-ride
// the default ones in the JSXMLDocumentHandler Object. A
// useful implementation would use these event functions to
// populate an object model, or do some other type of data capturing.
function handleElementStart(name)
{
	alert("ElementStart: " + name);
}

function handleElementEnd(name)
{
	alert("ElementEnd: " + name);
}

function handleDocumentStart()
{
	alert("DocumentStart");
}

function handleDocumentEnd()
{
	alert("DocumentEnd");
}

function handleText(value)
{
	alert("Text: " + value);
}

function handleAttribute(name, value)
{
	alert("Attribute: " + name + ", " + value);
}

// It's best to define a function that gets called
// after the page has completely loaded in order to
// instantiate the JSXMLDocumentHandler and JSXMLParser
// Objects, since the JS file gets loaded in the <body>
// section of the document.
function initPage()
{
	// Create a new JSXMLDocumentHandler Object.
	docHandler = new JSXMLDocumentHandler();
	
	// Over-ride the document handler's default
	// event handling functions by assigning them
	// to the functions defined above.
	docHandler.documentStart = handleDocumentStart;
	docHandler.documentEnd = handleDocumentEnd;
	docHandler.elementStart = handleElementStart;
	docHandler.elementEnd = handleElementEnd;
	docHandler.text = handleText;
	docHandler.attribute = handleAttribute;

	// Create an instance of the parser.
	parser = new JSXMLParser(docHandler,errHandler);
	
	// Transfer execution to a function designed to
	// wait until the parser is ready before it proceeds.
	waitForParser();
}

// waitForParser() will continue to call itself, via
// a setTimeout() call, until the parser has completed
// initializing itslef, at which point it calls the
// parseFile function.
function waitForParser()
{
	if (parser.parserReady())
	{
		// Tell the parser to load the specified file
		// from the server and parse it. Once parsing begins
		// events will be sent to docHandler's event handling
		// functions. 
		parser.parseFile(FILENAME);
	}
	else
	{
		setTimeout("waitForParser()",250);
	}
}
</script>
</head>
<body onLoad="initPage()">
<script src="jsxmlparser.js"></script>
</body>
</html>

myfile.xml:
-----------
<?xml version="1.0" encoding="iso-8859-1"?>
<people>
	<person age="23" height="5'9&quot;" weight="180" hair="brown">
		<name>Bart</name>
	</person>
	<person age="78" height="6'0&quot;" weight="220" hair="blonde">
		<name>Charlie</name>
	</person>
	<person age="3" height="2'9&quot;" weight="35" hair="red">
		<name>Tommy</name>
	</person>
</people>

*/

// The location (URL) of the TextFetcher.jar file that contains the Java classes
// necessary to load files from the server (relative to the HTML file that this
// JS file is included in).
var TEXT_FETCHER_JAR_LOCATION = "jars/TextFetcher.jar";

// Variable used to determine how XML documents will be loaded.  The XML Island will be
// used for IE 5+, and all other browsers will use a Java applet.
var USING_ISLAND = (navigator.appName == "Microsoft Internet Explorer" && parseInt(navigator.appVersion.slice(navigator.appVersion.indexOf("MSIE ")+5,navigator.appVersion.length)) >= 5);

// Write either the XML Island tag or the applet tag to the document.
if (USING_ISLAND)
{
	document.writeln("<xml id='xmlisland'></xml>");
}
else
{
	document.writeln("<applet name='TextFetcher' code=TextFetcher.class archive='" + TEXT_FETCHER_JAR_LOCATION + "' width=0 height=0></applet>");
}

//****************************************************************************
//
// checkApplet()
// Function that is used to detect when the TextFetcher applet has loaded.
//
// checkApplet() checks for the existence of the applet named "TextFetcher".
// The TextFetcher applet is used to read an XML file from the server. The
// global variable _AppletReady, which is initialized to false, will get set
// to true when the applet has been detected (the JSXMLParser Object uses the
// _AppletReady variable in its parserReady() member function).
//
// Input Params: None
//
// Output Params: None
//
// Returns: None
//
// Calls/References: variable _AppletReady
//
//****************************************************************************
var _AppletReady = false;
function checkApplet()
{
	if (document.applets["TextFetcher"])
	{
		_AppletReady = true;
	}
	else
	{
		setTimeout("checkApplet()",250);
	}
}

//****************************************************************************
//
// JSXMLDocumentHandler()
// The constructor for the JSXMLDocumentHandler Object.
//
// The JSXMLDocumentHandler Object is an event object used by the JSXMLParser
// Object. It's purpose is to provide an interface for relaying SAX-like
// document events generated by the JSXMLParser's parseString() and parseFile()
// functions.
// 
// Input Params: None
//
// Output Params: None
//
// Returns:
// A JSXMLDocumentHandler Object
//
// Calls/References: None
//
//****************************************************************************
//
// Member Functions:
// Null documentStart(Null)
//   Gets called when parsing of an XML document begins.
// Null documentEnd(Null)
//   Gets called when parsing of an XML document ends.
// Null elementStart(String name)
//   Gets called when parsing of an XML document element begins.
// Null elementEnd(String name)
//   Gets called when parsing of an XML document element ends.
// Null attribute(String name, String value)
//   Gets called when parsing an XML document element's attribute.
// Null text(String value)
//   Gets called when parsing an XML document element's text node.
//
//****************************************************************************
function JSXMLDocumentHandler()
{
}
// Setup default error handlers
JSXMLDocumentHandler.prototype.documentStart = function () {};
JSXMLDocumentHandler.prototype.documentEnd = function () {};
JSXMLDocumentHandler.prototype.elementStart = function (name) {};
JSXMLDocumentHandler.prototype.elementEnd = function (name) {};
JSXMLDocumentHandler.prototype.attribute = function (name, value) {};
JSXMLDocumentHandler.prototype.text = function (value) {};

//****************************************************************************
//
// JSXMLParser()
// The constructor for the JSXMLParser Object.
//
// The JSXMLParser Object functions as a simple DOM & SAX-type parser.  The JSXMLParser
// Object uses the tinyxmldom (a JavaScript XML DOM parser) as the actual parser.
// Retrieving of files from the server is done one of two ways, depending on the
// browser being used: the XML Island is used for IE 5+; all others use an applet.
// 
// Input Params:     
// documentHandler (Optional) - The JSXMLDocumentHandler that document events are
//                              to be delivered to. If not specified, then the 
//                              default document handler will be used.  NOTE: the
//                              default document handler does nothing in response
//                              to document events - it simply catches them and
//                              returns without any further processing.
// errorHandler (Optional) - The Function that should be called when parsing errors
//                           occur. If not specified, then the default error handler
//                           will be used.  NOTE: the default error handler does 
//                           nothing in response to error events - it simply catches
//                           them and returns without any further processing.
//
// Output Params: None
//
// Returns:
// A JSXMLParser Object
//
// Calls/References:
// JSXMLDocumentHandler - the default document handler Object.
//
//****************************************************************************
//
// Member Functions:
// XMLDoc parseFile(String filepath)
//   Parses the specified file, sending all document events to the specified
//   document handler, and all error events to the specified error handler.
//   Returns an XMLDoc object (see the tinyxmldom.js file for details on the
//   XMLDoc properties/methods).
// XMLDoc parseString(String xmlText)
//   Parses the specified XML text, sending all document events to the specified
//   document handler, and all error events to the specified error handler.
//   Returns an XMLDoc object (see the tinyxmldom.js file for details on the
//   XMLDoc properties/methods).
// Boolean parserReady()
//   Returns true if the parser is ready to be used, false otherwise. The caller
//   should use this function to ensure the parser's ready state before making any
//   calls to parseFile() or parseString().
//
//****************************************************************************
function JSXMLParser(documentHandler, errorHandler)
{
	if (documentHandler != null)
	{
		this.documentHandler = documentHandler;	
	}
	if (errorHandler != null)
	{
		this.errorHandler = errorHandler;	
	}
	this.parseFile = JSXMLParserParseFile;
	this.parseString = JSXMLParserParseString;
	this.parseRec = JSXMLParserParseRec;
	this.parserReady = function () {return _AppletReady};
	
	if (USING_ISLAND)
	{
		// If the ispland is being used, then we don't need to
		// wait for the applet to load.
		_AppletReady = true;
	}
	else
	{
		// Start the applet checking routine.
		checkApplet();
	}

	return this;	
}
// Setup default documentHandler and errorHandler properties.
JSXMLParser.prototype.documentHandler = new JSXMLDocumentHandler();
JSXMLParser.prototype.errorHandler = function () {};

//****************************************************************************
//
// JSXMLParserParseFile()
// Function that is used by the JSXMLParser Object to parse an XML file.
//
// Input Params: filepath - String - path of the XML file to be read, relative
//                          to the location of the HTML document that this JS
//                          file is loaded in.
//
// Output Params: None
//
// Returns: XMLDoc if successful, null otherwise.
//
// Calls/References: variable _AppletReady
//
//****************************************************************************
function JSXMLParserParseFile(filepath)
{
	var xmlText = "";
	
	if (this.parserReady())
	{
		// Load the document with either
		// the XML Island or the applet.
		if (USING_ISLAND)
		{
			xmlisland.async = false;
			xmlisland.load(filepath);
			if (xmlisland.parseError.errorCode == 0)
			{
				xmlText = xmlisland.xml + "";
			}
			else
			{
				xmlText = "";
				this.errorHandler(xmlisland.parseError.reason);
				return null;
			}
		}
		else
		{
			xmlText = document.applets['TextFetcher'].getTextRelativePath(filepath);
			xmlText = xmlText + "";
			// If xmlText != "" then the load was successful, and
			// it's contents (xmlText) will be parsed; otherwise
			// an error handler is invoked.
			if (xmlText == "")
			{
				this.errorHandler("Couldn't load the specified file: " + filepath);
				return null;
			}
		}
		return this.parseString(xmlText);
	}
	else
	{
		this.errorHandler("Parser not ready!");
		return null;
	}
}

//****************************************************************************
//
// JSXMLParserParseString()
// Function that is used by the JSXMLParser Object to parse an XML String.
//
// Input Params: xmlText - String - holds the XML mark-up to be parsed.
//
// Output Params: None
//
// Returns: XMLDoc if successful, null otherwise.
//
// Calls/References: None
//
//****************************************************************************
function JSXMLParserParseString(xmlText)
{
	
	var objDom = new XMLDoc(xmlText,this.errorHandler);
	var root = objDom.docNode;
	this.documentHandler.documentStart();
	this.parseRec(root);
	this.documentHandler.documentEnd();
	return objDom;
}

//****************************************************************************
//
// JSXMLParserParseRec()
// Recursive function that is used by the JSXMLParser Object to process an 
// XML document and fire SAX-like document events.
//
// Input Params: node - XMLNode - the node to be parsed.
//
// Output Params: None
//
// Returns: None
//
// Calls/References: trimWhitespace(), replaceEntities()
//
//****************************************************************************
function JSXMLParserParseRec(node)
{
	if (node.nodeType == "ELEMENT")
	{
		this.documentHandler.elementStart(node.tagName);
		
		// attributes
		var attrArr = node.getAttributeNames();
		for (var i=0;i<attrArr.length;i++)
		{
			this.documentHandler.attribute(attrArr[i], replaceEntities(trimWhitespace(node.getAttribute(attrArr[i]))));
		}
		
		var elemArr = node.children;
		if (elemArr)
		{
			for (var i=0;i<elemArr.length;i++)
			{
				this.parseRec(elemArr[i]);
			}
		}
		this.documentHandler.elementEnd(node.tagName);
	}
	else if (node.nodeType == "TEXT")
	{
		var data = trimWhitespace(node.getText());
		if (data.length > 0)
		{
			this.documentHandler.text(data);
		}
	}
	else if (node.nodeType == "CDATA")
	{
		var data = trimWhitespace(node.getText());
		if (data.length > 0)
		{
			this.documentHandler.text(data);
		}
	}
}

//****************************************************************************
//
// trimWhitespace()
// Helper function that trims whitespace from the begining and end of a String.
//
// Input Params: strVal - String - the String to be trimmed.
//
// Output Params: None
//
// Returns: String - the trimmed version of strVal.
//
// Calls/References: None
//
//****************************************************************************
function trimWhitespace(strVal)
{
	if (strVal == null) strVal = "";
	var strTemp = "";
	var charCode = 0;
	// Search, from the begining of strVal to the end, until a non-whitespace
	// character is encountered; save everything from that position to the end
	// in the variable strTemp and break out of the loop.
	for (var i=0;i<strVal.length;i++)
	{
		charCode = strVal.charCodeAt(i);
		if (charCode > 32)
		{
			strTemp = strVal.slice(i,strVal.length);
			break;
		}
	}
	
	// Search, from the end of strTemp to the begining, until a non-whitespace character
	// is encountered; save everything from that position to the begining in strTemp and
	// break out of the loop.
	for (var i=strTemp.length-1;i>-1;i--)
	{
		charCode = strVal.charCodeAt(i);
		if (charCode > 32)
		{
			strTemp = strTemp.slice(0,i+1);
			break;
		}
	}
	return strTemp;
}

//****************************************************************************
//
// replaceEntities()
// Helper function that replaces common entities with their literal values.
//
// Input Params: strVal - String - the String to be processed.
//
// Output Params: None
//
// Returns: String - a copy of strVal with all entities replace with their
//          literal values.
//
// Calls/References: None
//
//****************************************************************************
function replaceEntities(strVal)
{
	var retVal = strVal + "";
	retVal = retVal.replace("&quot;","\"");
	retVal = retVal.replace("&amp;","&");
	retVal = retVal.replace("&gt;",">");
	retVal = retVal.replace("&lt;","<");
	retVal = retVal.replace("&apos;","'");
	
	return retVal;
}